/* SPDX-License-Identifier: GPL-2.0-only */

#include <libpayload.h>
#include <tests/test.h>

#include "../liblz4/lz4_wrapper.c"

/* This macro basically does nothing but suppresses linter messages */
#define EMPTY_WRAP(...) __VA_ARGS__

#define TEST_RAW_DATA_SIZE sizeof((u8[]){TEST_RAW_DATA})
/* TEST_RAW_DATA="LLLLLLLLLLLLLLLLZZZZZZZZZZZZZZZZ4444444444444444"
   echo ${TEST_RAW_DATA} | hexdump -e '1/1 "0x%02x" ", "' -v */
#define TEST_RAW_DATA EMPTY_WRAP( \
	0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, \
	0x4c, 0x4c, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, 0x5a, \
	0x5a, 0x5a, 0x5a, 0x5a, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, \
	0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x0a)

#define TEST_LZ4_DATA_SIZE sizeof((u8[]){TEST_LZ4_DATA})
/* echo ${TEST_RAW_DATA} | lz4 -12 --favor-decSpeed -c | \
   hexdump -e '1/1 "0x%02x" ", "' -v */
#define TEST_LZ4_DATA EMPTY_WRAP( \
	0x04, 0x22, 0x4d, 0x18, 0x64, 0x40, 0xa7, 0x15, 0x00, 0x00, 0x00, 0x2a, 0x4c, 0x4c, \
	0x02, 0x00, 0x2a, 0x5a, 0x5a, 0x02, 0x00, 0x26, 0x34, 0x34, 0x02, 0x00, 0x50, 0x34, \
	0x34, 0x34, 0x34, 0x0a, 0x00, 0x00, 0x00, 0x00, 0xe1, 0xea, 0x87, 0xfa)

#define TEST_LEGACY_LZ4_DATA_SIZE sizeof((u8[]){TEST_LEGACY_LZ4_DATA})
/* echo ${TEST_RAW_DATA} | lz4 -l -12 --favor-decSpeed -c | \
   hexdump -e '1/1 "0x%02x" ", "' -v */
#define TEST_LEGACY_LZ4_DATA EMPTY_WRAP( \
	0x02, 0x21, 0x4c, 0x18, 0x12, 0x00, 0x00, 0x00, 0x1b, 0x4c, 0x01, 0x00, 0x1b, 0x5a, \
	0x01, 0x00, 0x17, 0x34, 0x01, 0x00, 0x50, 0x34, 0x34, 0x34, 0x34, 0x0a)

#define TEST_LARGE_RAW_DATA_SIZE sizeof((u8[]){TEST_LARGE_RAW_DATA})
/*
 * LINUX_IMG_PATH=<set to available uncompressed ARM Linux Image>;
 * hexdump -e '1/1 "0x%02x" ", "' ${LINUX_IMG_PATH} -n 320 -v
 * Arm Linux kernel image header is 64 bytes long. Size of scratch buffer in one of the
 * payloads is 256 bytes + Arm Linux kernel image header. Hence 320 bytes.
 */
#define TEST_LARGE_RAW_DATA EMPTY_WRAP( \
	0x4d, 0x5a, 0x40, 0xfa, 0x19, 0x00, 0x7f, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x72, 0x02, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x41, 0x52, 0x4d, 0x64, 0x40, 0x00, 0x00, 0x00, 0x50, 0x45, 0x00, 0x00, 0x64, 0xaa, \
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0xa0, 0x00, 0x06, 0x02, 0x0b, 0x02, 0x02, 0x14, 0x00, 0x00, 0x3b, 0x02, 0x00, 0x00, \
	0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x03, 0x02, 0x00, 0x00, 0x01, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x0a, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2e, 0x74, 0x65, 0x78, \
	0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3b, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, \
	0x3b, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x60, 0x2e, 0x64, 0x61, 0x74, 0x61, 0x00, \
	0x00, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x3c, 0x02, 0x00, 0x0a, 0x2b, 0x00, \
	0x00, 0x00, 0x3c, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00)

#define TEST_LARGE_LZ4_DATA_SIZE sizeof((u8[]){TEST_LARGE_LZ4_DATA})
/* head -c 320 ${LINUX_IMG_PATH} | lz4 -12 --favor-decSpeed -c | \
   hexdump -e '1/1 "0x%02x" ", "' -v */
#define TEST_LARGE_LZ4_DATA EMPTY_WRAP( \
	0x04, 0x22, 0x4d, 0x18, 0x64, 0x40, 0xa7, 0x9a, 0x00, 0x00, 0x00, 0xa4, 0x4d, 0x5a, \
	0x40, 0xfa, 0x19, 0x00, 0x7f, 0x14, 0x00, 0x00, 0x02, 0x00, 0x20, 0x72, 0x02, 0x0c, \
	0x00, 0x3e, 0x0a, 0x00, 0x00, 0x02, 0x00, 0x07, 0x14, 0x00, 0xf9, 0x00, 0x41, 0x52, \
	0x4d, 0x64, 0x40, 0x00, 0x00, 0x00, 0x50, 0x45, 0x00, 0x00, 0x64, 0xaa, 0x02, 0x1c, \
	0x00, 0xf1, 0x00, 0xa0, 0x00, 0x06, 0x02, 0x0b, 0x02, 0x02, 0x14, 0x00, 0x00, 0x3b, \
	0x02, 0x00, 0x00, 0x36, 0x14, 0x00, 0x77, 0x04, 0x04, 0x03, 0x02, 0x00, 0x00, 0x01, \
	0x26, 0x00, 0x33, 0x01, 0x00, 0x00, 0x37, 0x00, 0x19, 0x03, 0x3e, 0x00, 0x15, 0x72, \
	0x28, 0x00, 0x19, 0x0a, 0x31, 0x00, 0x0e, 0x0a, 0x00, 0x04, 0x1c, 0x00, 0x3f, 0x06, \
	0x00, 0x00, 0x02, 0x00, 0x1e, 0x51, 0x2e, 0x74, 0x65, 0x78, 0x74, 0x0a, 0x00, 0x13, \
	0x3b, 0x70, 0x00, 0x2c, 0x3b, 0x02, 0x6f, 0x00, 0x90, 0x20, 0x00, 0x00, 0x60, 0x2e, \
	0x64, 0x61, 0x74, 0x61, 0x0e, 0x00, 0x01, 0xc8, 0x00, 0x52, 0x3c, 0x02, 0x00, 0x0a, \
	0x2b, 0x08, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00, 0xf6, 0xb4, 0xb7, 0x4f)

#define TEST_LARGE_LEGACY_LZ4_DATA_SIZE sizeof((u8[]){TEST_LARGE_LEGACY_LZ4_DATA})
/* head -c 320 ${LINUX_IMG_PATH} | lz4 -l -12 --favor-decSpeed -c | \
   hexdump -e '1/1 "0x%02x" ", "' -v */
#define TEST_LARGE_LEGACY_LZ4_DATA EMPTY_WRAP( \
	0x02, 0x21, 0x4c, 0x18, 0x93, 0x00, 0x00, 0x00, 0x95, 0x4d, 0x5a, 0x40, 0xfa, 0x19, \
	0x00, 0x7f, 0x14, 0x00, 0x01, 0x00, 0x20, 0x72, 0x02, 0x06, 0x00, 0x2f, 0x0a, 0x00, \
	0x01, 0x00, 0x0b, 0xf9, 0x00, 0x41, 0x52, 0x4d, 0x64, 0x40, 0x00, 0x00, 0x00, 0x50, \
	0x45, 0x00, 0x00, 0x64, 0xaa, 0x02, 0x1c, 0x00, 0xf1, 0x00, 0xa0, 0x00, 0x06, 0x02, \
	0x0b, 0x02, 0x02, 0x14, 0x00, 0x00, 0x3b, 0x02, 0x00, 0x00, 0x36, 0x14, 0x00, 0x76, \
	0x04, 0x04, 0x03, 0x02, 0x00, 0x00, 0x01, 0x26, 0x00, 0x00, 0x0c, 0x00, 0x03, 0x37, \
	0x00, 0x19, 0x03, 0x3e, 0x00, 0x15, 0x72, 0x28, 0x00, 0x1a, 0x0a, 0x31, 0x00, 0x0f, \
	0x01, 0x00, 0x06, 0x2f, 0x06, 0x00, 0x01, 0x00, 0x1f, 0x51, 0x2e, 0x74, 0x65, 0x78, \
	0x74, 0x0a, 0x00, 0x13, 0x3b, 0x70, 0x00, 0x2c, 0x3b, 0x02, 0x6f, 0x00, 0x91, 0x20, \
	0x00, 0x00, 0x60, 0x2e, 0x64, 0x61, 0x74, 0x61, 0x0e, 0x00, 0x00, 0xc8, 0x00, 0x52, \
	0x3c, 0x02, 0x00, 0x0a, 0x2b, 0x08, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
	0x00)

static void test_lz4(void **state)
{
	const u8 test_src_data[TEST_LZ4_DATA_SIZE] = { TEST_LZ4_DATA };
	u8 test_dst_data[TEST_RAW_DATA_SIZE] = {0};
	u8 test_uncompressed_data[TEST_RAW_DATA_SIZE] = { TEST_RAW_DATA };
	size_t dst_size;

	dst_size = ulz4fn(test_src_data, sizeof(test_src_data),
			  test_dst_data, sizeof(test_dst_data));
	assert_int_equal(TEST_RAW_DATA_SIZE, dst_size);
	assert_memory_equal(test_dst_data, test_uncompressed_data, TEST_RAW_DATA_SIZE);
}

static void test_lz4_partial_decompression(void **state)
{
	const u8 test_src_data[TEST_LARGE_LZ4_DATA_SIZE] = { TEST_LARGE_LZ4_DATA };
	u8 test_dst_data[TEST_LARGE_RAW_DATA_SIZE / 2] = {0};
	u8 test_uncompressed_data[TEST_LARGE_RAW_DATA_SIZE] = { TEST_LARGE_RAW_DATA };
	size_t dst_size;

	ulz4fn(test_src_data, sizeof(test_src_data),
			  test_dst_data, sizeof(test_dst_data));
	for (dst_size = 0; dst_size < sizeof(test_dst_data); dst_size++) {
		if (test_dst_data[dst_size] != test_uncompressed_data[dst_size])
			break;
	}
	assert_in_range(dst_size, TEST_LARGE_RAW_DATA_SIZE / 4, TEST_LARGE_RAW_DATA_SIZE);
}

static void test_legacy_lz4(void **state)
{
	const u8 test_src_data[TEST_LEGACY_LZ4_DATA_SIZE] = { TEST_LEGACY_LZ4_DATA };
	u8 test_dst_data[TEST_RAW_DATA_SIZE] = {0};
	u8 test_uncompressed_data[TEST_RAW_DATA_SIZE] = { TEST_RAW_DATA };
	size_t dst_size;

	dst_size = ulz4ln(test_src_data, sizeof(test_src_data),
			  test_dst_data, sizeof(test_dst_data));
	assert_int_equal(TEST_RAW_DATA_SIZE, dst_size);
	assert_memory_equal(test_dst_data, test_uncompressed_data, TEST_RAW_DATA_SIZE);
}

static void test_legacy_lz4_partial_decompression(void **state)
{
	const u8 test_src_data[TEST_LARGE_LEGACY_LZ4_DATA_SIZE] = { TEST_LARGE_LEGACY_LZ4_DATA };
	u8 test_dst_data[TEST_LARGE_RAW_DATA_SIZE / 2] = {0};
	u8 test_uncompressed_data[TEST_LARGE_RAW_DATA_SIZE] = { TEST_LARGE_RAW_DATA };
	size_t dst_size;

	ulz4ln(test_src_data, sizeof(test_src_data),
			  test_dst_data, sizeof(test_dst_data));
	for (dst_size = 0; dst_size < sizeof(test_dst_data); dst_size++) {
		if (test_dst_data[dst_size] != test_uncompressed_data[dst_size])
			break;
	}
	assert_in_range(dst_size, TEST_LARGE_RAW_DATA_SIZE / 4, TEST_LARGE_RAW_DATA_SIZE);
}

int main(void)
{
	const struct CMUnitTest tests[] = {
		cmocka_unit_test(test_lz4),
		cmocka_unit_test(test_lz4_partial_decompression),
		cmocka_unit_test(test_legacy_lz4),
		cmocka_unit_test(test_legacy_lz4_partial_decompression),
	};

	return lp_run_group_tests(tests, NULL, NULL);
}
